<?php

namespace App\Http\Controllers;

use App\Models\ComputerProduct;
use Illuminate\Http\Request;
use DB;

class StockController extends Controller
{
    public function index(Request $request)
    {
        $branchId = auth()->user()->branch_id;
        $search = trim((string) $request->query('search'));
        $isNumericSearch = is_numeric($search);

        $movementSummary = DB::table('stock_movements')
            ->select('variation_id', DB::raw('SUM(qty) as quantity'))
            ->where('branch_id', $branchId)
            ->groupBy('variation_id');

        $baseStockQuery = DB::table('product_variations as pv')
            ->select(
                'pv.id as variation_id',
                'pv.sku',
                'pv.color',
                'pv.storage',
                'p.name as product_name',
                'pv.serial_number',
                'p.id as product_id',
                DB::raw('COALESCE(sm.quantity, 0) as quantity')
            )
            ->join('products as p', 'p.id', '=', 'pv.product_id')
            ->leftJoinSub($movementSummary, 'sm', function ($join) {
                $join->on('sm.variation_id', '=', 'pv.id');
            })
            ->orderBy('p.name')
            ->orderBy('pv.id');

        $phoneSuggestionSource = (clone $baseStockQuery)->get();

        $stock = (clone $baseStockQuery)
            ->when($search !== '', function ($query) use ($search, $isNumericSearch) {
                $like = '%' . $search . '%';

                $query->where(function ($inner) use ($like, $isNumericSearch, $search) {
                    $inner->where('p.name', 'like', $like)
                        ->orWhere('pv.serial_number', 'like', $like)
                        ->orWhere('pv.sku', 'like', $like);

                    if ($isNumericSearch) {
                        $inner->orWhere('pv.id', (int) $search)
                            ->orWhere('p.id', (int) $search);
                    }
                });
            })
            ->get();

        $computerQuery = ComputerProduct::orderBy('name');

        $computerStock = (clone $computerQuery)
            ->when($search !== '', function ($query) use ($search) {
                $like = '%' . $search . '%';

                $query->where(function ($inner) use ($like, $search) {
                    $inner->where('name', 'like', $like)
                        ->orWhere('serial_number', 'like', $like)
                        ->orWhere('product_code', 'like', $like);
                });
            })
            ->get();

        $computerSuggestionSource = $search !== ''
            ? (clone $computerQuery)->get()
            : $computerStock;

        $searchSuggestions = collect()
            ->merge($phoneSuggestionSource->flatMap(function ($item) {
                return collect([
                    [
                        'value' => $item->product_name,
                        'label' => 'Phone · ' . $item->product_name,
                    ],
                    [
                        'value' => $item->serial_number,
                        'label' => 'Phone Serial · ' . $item->serial_number,
                    ],
                    [
                        'value' => (string) $item->product_id,
                        'label' => 'Phone Product ID · ' . $item->product_id,
                    ],
                ]);
            }))
            ->merge($computerSuggestionSource->flatMap(function ($item) {
                return collect([
                    [
                        'value' => $item->name,
                        'label' => 'Computer · ' . $item->name,
                    ],
                    [
                        'value' => $item->serial_number,
                        'label' => 'Computer Serial · ' . $item->serial_number,
                    ],
                    [
                        'value' => $item->product_code,
                        'label' => 'Computer Product ID · ' . $item->product_code,
                    ],
                ]);
            }))
            ->filter(function ($option) {
                return isset($option['value']) && trim((string) $option['value']) !== '';
            })
            ->unique('value')
            ->take(50)
            ->values()
            ->all();

        return view('stock.index', [
            'stock' => $stock,
            'computerStock' => $computerStock,
            'searchTerm' => $search,
            'searchSuggestions' => $searchSuggestions,
        ]);
    }
}
