<?php

namespace App\Http\Controllers;

use App\Helpers\NumberHelper;
use App\Models\ProductVariation;
use App\Models\ComputerProduct;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class SalesController extends Controller
{
    protected array $paymentMethods = [
        'ABA',
        'AC',
        'Wing',
        'Bank',
        'Cash',
        'Bank + Cash',
        'Cash with delivery',
    ];

    public function paymentMethods(): array
    {
        return $this->paymentMethods;
    }

    public function index()
    {
        $sales = Sale::with('branch')
            ->withSum('items as total_qty', 'qty')
            ->latest()
            ->paginate(20);
        return view('sales.index', compact('sales'));
    }

    public function create()
    {
        $variations = ProductVariation::with('product')->orderBy('sku')->get();
        $computerProducts = ComputerProduct::orderBy('name')->get();
        $paymentMethods = $this->paymentMethods;
        $saleNumber = NumberHelper::generateTimestampCode('SL');
        return view('sales.create', compact('variations', 'computerProducts', 'paymentMethods', 'saleNumber'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'sale_date' => 'required|date',
            'customer_id' => 'nullable|integer|min:1',
            'payment_method' => ['required', Rule::in($this->paymentMethods)],
            'customer_phone' => 'nullable|string|max:30',
            'amount' => 'nullable|numeric|min:0',
            'profit' => 'nullable|numeric|min:0',
            'paid_time' => 'nullable|date_format:H:i',
            'other' => 'nullable|string|max:500',
            'tax' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'bank_amount' => 'nullable|numeric|min:0',
            'cash_amount' => 'nullable|numeric|min:0',
            'items' => 'required|array|min:1',
            'items.*.product_type' => ['required', Rule::in(['phone', 'computer'])],
            'items.*.variation_id' => 'nullable|exists:product_variations,id',
            'items.*.computer_product_id' => 'nullable|exists:computer_products,id',
            'items.*.qty' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
        ]);

        foreach ($validated['items'] as $index => $item) {
            if ($item['product_type'] === 'phone' && empty($item['variation_id'])) {
                return back()->withErrors(["items.$index.variation_id" => 'Please select a phone variation.'])->withInput();
            }

            if ($item['product_type'] === 'computer' && empty($item['computer_product_id'])) {
                return back()->withErrors(["items.$index.computer_product_id" => 'Please select a computer product.'])->withInput();
            }
        }

        $totalAmount = collect($validated['items'])->reduce(function ($carry, $item) {
            return $carry + ($item['qty'] * $item['price']);
        }, 0);

        $autoCustomerId = (int) now()->format('ymdHis');
        $customerId = $request->filled('customer_id') ? (int) $request->customer_id : $autoCustomerId;

        $invoiceNumber = NumberHelper::generateTimestampCode('SL');

        $taxValue = isset($validated['tax']) ? (float) $validated['tax'] : 0;
        $discountValue = isset($validated['discount']) ? (float) $validated['discount'] : 0;
        $maxDiscount = max(0, $totalAmount + $taxValue);
        $discountValue = min($discountValue, $maxDiscount);
        $amountValue = $request->filled('amount')
            ? (float) $validated['amount']
            : max(0, $totalAmount + $taxValue - $discountValue);

        $bankAmountInput = $request->input('bank_amount');
        $cashAmountInput = $request->input('cash_amount');

        $bankAmount = ($bankAmountInput === null || $bankAmountInput === '')
            ? null
            : (float) $bankAmountInput;
        $cashAmount = ($cashAmountInput === null || $cashAmountInput === '')
            ? null
            : (float) $cashAmountInput;

        if ($validated['payment_method'] === 'Bank + Cash') {
            if ($bankAmount === null || $cashAmount === null) {
                return back()->withErrors([
                    'bank_amount' => 'Please enter both bank and cash amounts for split payments.'
                ])->withInput();
            }

            $splitTotal = round($bankAmount + $cashAmount, 2);
            $expectedTotal = round($amountValue, 2);

            if (abs($splitTotal - $expectedTotal) > 0.01) {
                return back()->withErrors([
                    'bank_amount' => 'Bank + Cash split must equal the invoice total (' . number_format($amountValue, 2) . ').'
                ])->withInput();
            }
        } else {
            $bankAmount = null;
            $cashAmount = null;
        }

        $invoice = Sale::create([
            'invoice_number' => $invoiceNumber,
            'customer_id' => $customerId,
            'customer_phone' => $validated['customer_phone'] ?? null,
            'branch_id' => auth()->user()->branch_id,
            'sale_date' => $validated['sale_date'],
            'total_amount' => $totalAmount,
            'amount' => $amountValue,
            'profit' => $validated['profit'] ?? null,
            'paid_time' => $validated['paid_time'] ?? null,
            'other' => $validated['other'] ?? null,
            'tax' => $taxValue,
            'discount' => $discountValue,
            'bank_amount' => $bankAmount,
            'cash_amount' => $cashAmount,
            'payment_method' => $validated['payment_method'],
            'status' => 'paid',
        ]);

        foreach ($validated['items'] as $item) {
            if ($item['product_type'] === 'computer') {
                $computer = ComputerProduct::findOrFail($item['computer_product_id']);
                $serial = $computer->serial_number;

                SaleItem::create([
                    'sale_id' => $invoice->id,
                    'product_type' => 'computer',
                    'computer_product_id' => $computer->id,
                    'variation_id' => null,
                    'serial_number' => $serial,
                    'qty' => $item['qty'],
                    'price' => $item['price'],
                ]);

                $computer->decrement('stock_qty', $item['qty']);
            } else {
                $variation = ProductVariation::findOrFail($item['variation_id']);

                SaleItem::create([
                    'sale_id' => $invoice->id,
                    'product_type' => 'phone',
                    'variation_id' => $variation->id,
                    'serial_number' => $variation->serial_number ?? null,
                    'qty' => $item['qty'],
                    'price' => $item['price'],
                ]);

                StockMovement::create([
                    'branch_id' => $invoice->branch_id,
                    'variation_id' => $variation->id,
                    'type' => 'sale',
                    'qty' => -$item['qty'],
                    'reference' => $invoice->invoice_number,
                ]);
            }
        }

        return redirect()->route('sales.show', $invoice->id);
    }

    public function show($id)
    {
        $sale = Sale::with(['items.variation.product', 'items.computerProduct', 'branch'])
            ->withSum('items as total_qty', 'qty')
            ->findOrFail($id);
        return view('sales.show', compact('sale'));
    }
}
