<?php

use Illuminate\Support\Facades\Route;

use App\Http\Controllers\DashboardController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\VariationController;
use App\Http\Controllers\PurchaseController;
use App\Http\Controllers\SalesController;
use App\Http\Controllers\StockController;
use App\Http\Controllers\RepairController;
use App\Http\Controllers\ExpenseController;
use App\Http\Controllers\SupplierController;
use App\Http\Controllers\BranchController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\RoleController;

// -------------------------------
// AUTH MIDDLEWARE
// -------------------------------
Route::middleware(['auth'])->group(function () {

    // Dashboard (BI)
    Route::get('/', [DashboardController::class, 'index'])->name('dashboard');

    // ---------------------------
    // BRANCH SWITCHING
    // ---------------------------
    Route::post('/branch/switch/{id}', [BranchController::class, 'switch'])->name('branch.switch');

    Route::get('/profile', [ProfileController::class, 'show'])->name('profile.show');
    Route::get('/profile/edit', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');

    Route::get('/settings', [SettingsController::class, 'index'])->name('settings.index');

    Route::middleware('role:Admin')->prefix('settings')->name('settings.')->group(function () {
        Route::resource('roles', RoleController::class)->except(['show', 'create']);
        Route::get('users', [SettingsController::class, 'users'])->name('users.index');
        Route::post('users', [SettingsController::class, 'storeUser'])->name('users.store');
        Route::patch('users/{user}/roles', [SettingsController::class, 'updateUserRoles'])->name('users.roles.update');
    });


    // ===========================
    // PRODUCTS
    // ===========================
    Route::middleware(['role:Admin|Manager|Accountant'])->group(function () {
        Route::get('/computers/search', [\App\Http\Controllers\ComputerProductController::class, 'search'])->name('computers.search');
    });

    Route::middleware(['role:Admin|Manager'])->group(function () {
        Route::resource('products', ProductController::class);
        Route::post('/products/import', [ProductController::class, 'import'])->name('products.import');
        Route::get('/computers', [\App\Http\Controllers\ComputerProductController::class, 'index'])->name('computers.index');
        Route::get('/computers/create', [\App\Http\Controllers\ComputerProductController::class, 'create'])->name('computers.create');
        Route::get('/computers/{computer}', [\App\Http\Controllers\ComputerProductController::class, 'show'])
            ->whereNumber('computer')
            ->name('computers.show');
        Route::get('/computers/{computer}/edit', [\App\Http\Controllers\ComputerProductController::class, 'edit'])
            ->whereNumber('computer')
            ->name('computers.edit');
        Route::put('/computers/{computer}', [\App\Http\Controllers\ComputerProductController::class, 'update'])
            ->whereNumber('computer')
            ->name('computers.update');
        Route::delete('/computers/{computer}', [\App\Http\Controllers\ComputerProductController::class, 'destroy'])
            ->whereNumber('computer')
            ->name('computers.destroy');
        // Product variations
        Route::post('/variations/store', [VariationController::class, 'store'])
            ->name('variations.store');
    });

    // ===========================
    // PURCHASE ORDERS
    // ===========================
    Route::middleware(['role:Admin|Manager|Accountant'])->group(function () {

        Route::post('/computers', [\App\Http\Controllers\ComputerProductController::class, 'store'])->name('computers.store');
        Route::post('/computers/import', [\App\Http\Controllers\ComputerProductController::class, 'import'])->name('computers.import');
        Route::get('/purchases', [PurchaseController::class, 'index'])->name('purchases.index');
        Route::get('/purchases/create', [PurchaseController::class, 'create'])->name('purchases.create');
        Route::post('/purchases/store', [PurchaseController::class, 'store'])->name('purchases.store');
        Route::get('/purchases/{id}', [PurchaseController::class, 'show'])->name('purchases.show');

        // Receive Stock
        Route::post('/purchases/{id}/receive', [PurchaseController::class, 'receive'])
            ->name('purchases.receive');
    });

    // ===========================
    // SALES
    // ===========================
    Route::middleware(['role:Sales|Admin|Manager'])->group(function () {
        Route::resource('sales', SalesController::class);
    });

    // ===========================
    // STOCK
    // ===========================
    Route::get('/stock', [StockController::class, 'index'])->name('stock.index');

    // ===========================
    // REPAIRS
    // ===========================
    Route::middleware(['role:Technician|Manager|Admin'])->group(function () {
        Route::get('/repairs', [RepairController::class, 'index'])->name('repairs.index');
        Route::get('/repairs/create', [RepairController::class, 'create'])->name('repairs.create');
        Route::post('/repairs', [RepairController::class, 'store'])->name('repairs.store');
        Route::get('/repairs/{repair}', [RepairController::class, 'show'])->name('repairs.show');
        Route::get('/repairs/{repair}/edit', [RepairController::class, 'edit'])->name('repairs.edit');
        Route::put('/repairs/{repair}', [RepairController::class, 'update'])->name('repairs.update');
        Route::delete('/repairs/{repair}', [RepairController::class, 'destroy'])->name('repairs.destroy');
        Route::post('/repairs/{repair}/status', [RepairController::class, 'updateStatus'])->name('repairs.status');
    });

    // ===========================
    // EXPENSES
    // ===========================
    Route::middleware(['role:Accountant|Admin|Manager'])->group(function () {
        Route::get('/expenses', [ExpenseController::class, 'index'])->name('expenses.index');
        Route::post('/expenses/store', [ExpenseController::class, 'store'])->name('expenses.store');
    });

    // ===========================
    // SUPPLIERS
    // ===========================
    Route::middleware(['role:Admin|Manager'])->group(function () {
        Route::get('/suppliers', [SupplierController::class, 'index'])->name('suppliers.index');
        Route::post('/suppliers/store', [SupplierController::class, 'store'])->name('suppliers.store');
        Route::post('/suppliers/import', [SupplierController::class, 'import'])->name('suppliers.import');
    });
});

require __DIR__.'/auth.php';
