<?php

namespace App\Http\Controllers;

use App\Models\ProductVariation;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\StockMovement;
use App\Models\Supplier;
use App\Models\ComputerProduct;
use App\Helpers\NumberHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PurchaseController extends Controller
{
    public function index()
    {
        $purchases = Purchase::with('supplier')
            ->where('branch_id', auth()->user()->branch_id)
            ->latest()
            ->paginate(20);
        return view('purchases.index', compact('purchases'));
    }

    public function create()
    {
        $suppliers = Supplier::orderBy('name')->get();
        $variations = ProductVariation::with('product')->orderBy('sku')->get();
        $computerOptions = config('computers');
        $purchaseCode = NumberHelper::generateTimestampCode('PC');
        $sInvoiceNumber = NumberHelper::generateTimestampCode('SI', 'YmdHi');

        return view('purchases.create', compact(
            'suppliers',
            'variations',
            'computerOptions',
            'purchaseCode',
            'sInvoiceNumber'
        ));
    }

    public function store(Request $request)
    {
        $branchId = auth()->user()->branch_id;
        $purchaseType = $request->input('purchase_type');

        $rules = [
            'purchase_code' => ['required', 'string', 'max:255', Rule::unique('purchases', 'purchase_code')],
            'invoice_number' => 'required|string|max:255',
            's_invoice_number' => ['required', 'string', 'max:255', Rule::unique('purchases', 's_invoice_number')],
            'supplier_id' => 'required|exists:suppliers,id',
            'purchase_date' => 'required|date',
            'purchase_type' => ['required', Rule::in(['computer', 'phone'])],
            'tax_amount' => 'nullable|numeric|min:0',
            'other_charges' => 'nullable|numeric|min:0',
            'items' => 'required|array|min:1',
            'items.*.variation_id' => 'nullable|exists:product_variations,id',
            'items.*.computer_product_id' => 'nullable|exists:computer_products,id',
            'items.*.qty' => 'required|integer|min:1',
            'items.*.cost_price' => 'required|numeric|min:0.01',
            'items.*.serial_numbers' => 'required|array|min:1',
            'items.*.serial_numbers.*' => 'required|string|max:255',
        ];

        if ($purchaseType === 'computer') {
            $rules = array_merge($rules, [
                'items.*.computer_product_id' => ['required', 'exists:computer_products,id'],
            ]);
        }

        $validator = Validator::make($request->all(), $rules);
        $validator->after(function ($validator) use ($purchaseType) {
            $items = $validator->getData()['items'] ?? [];
            $allSerials = [];
            foreach ($items as $index => $item) {
                $qty = (int) ($item['qty'] ?? 0);
                $serials = array_values(array_filter($item['serial_numbers'] ?? [], fn ($serial) => $serial !== ''));

                if ($purchaseType === 'phone' && empty($item['variation_id'])) {
                    $validator->errors()->add("items.$index.variation_id", 'Please select a phone SKU.');
                }

                if ($purchaseType === 'computer' && empty($item['product_code'])) {
                    if (empty($item['computer_product_id'])) {
                        $validator->errors()->add("items.$index.computer_product_id", 'Product ID is required.');
                    }
                }

                if ($qty !== count($serials)) {
                    $validator->errors()->add("items.$index.serial_numbers", 'Serial number count must match quantity.');
                }

                if (count($serials) !== count(array_unique($serials))) {
                    $validator->errors()->add("items.$index.serial_numbers", 'Serial numbers must be unique per line item.');
                }

                foreach ($serials as $serial) {
                    $key = $purchaseType.'-'.$serial;
                    if (isset($allSerials[$key])) {
                        $validator->errors()->add("items.$index.serial_numbers", 'Serial numbers must be unique within the purchase order.');
                        break;
                    }
                    $allSerials[$key] = true;
                }

            }
        });

        $validated = $validator->validate();

        $totalAmount = collect($validated['items'])->reduce(function ($carry, $item) {
            return $carry + ($item['qty'] * $item['cost_price']);
        }, 0);
        $taxAmount = $validated['tax_amount'] ?? 0;
        $otherCharges = $validated['other_charges'] ?? 0;
        $totalDue = $totalAmount + $taxAmount + $otherCharges;

        $po = Purchase::create([
            'po_number' => NumberHelper::generate('PO', new Purchase),
            'purchase_code' => $validated['purchase_code'],
            'invoice_number' => $validated['invoice_number'] ?? null,
            's_invoice_number' => $validated['s_invoice_number'],
            'supplier_id' => $validated['supplier_id'],
            'branch_id' => $branchId,
            'purchase_date' => $validated['purchase_date'],
            'purchase_type' => $validated['purchase_type'],
            'total_amount' => $totalAmount,
            'tax_amount' => $taxAmount,
            'other_charges' => $otherCharges,
            'total_due' => $totalDue,
            'status' => 'pending'
        ]);

        foreach ($validated['items'] as $item) {
            $serials = array_values($item['serial_numbers'] ?? []);
            $qty = (int) $item['qty'];

            for ($index = 0; $index < $qty; $index++) {
                $serial = $serials[$index] ?? null;
                $computerId = null;

                if ($validated['purchase_type'] === 'computer') {
                    $computerId = $item['computer_product_id'];
                }

                PurchaseItem::create([
                    'purchase_id' => $po->id,
                    'product_type' => $validated['purchase_type'],
                    'variation_id' => $validated['purchase_type'] === 'phone' ? $item['variation_id'] : null,
                    'computer_product_id' => $validated['purchase_type'] === 'computer' ? $computerId : null,
                    'qty' => 1,
                    'cost_price' => $item['cost_price'],
                    'serial_number' => $serial,
                ]);
            }
        }

        return redirect()->route('purchases.show', $po->id);
    }

    public function show($id)
    {
        $purchase = Purchase::with(['items.variation.product', 'items.computerProduct'])->findOrFail($id);
        return view('purchases.show', compact('purchase'));
    }

    public function receive(Request $request, $id)
    {
        $purchase = Purchase::with('items.variation', 'items.computerProduct')->findOrFail($id);

        foreach ($purchase->items as $item) {
            if ($purchase->purchase_type === 'computer') {
                if ($item->computerProduct) {
                    $item->computerProduct->increment('stock_qty', $item->qty);
                }
                continue;
            }

            StockMovement::create([
                'branch_id' => auth()->user()->branch_id,
                'variation_id' => $item->variation_id,
                'type' => 'purchase',
                'qty' => $item->qty,
                'reference' => $purchase->po_number
            ]);
        }

        $purchase->update(['status' => 'received']);

        return back()->with('success', 'Items Received');
    }

}
