<?php

namespace App\Http\Controllers;

use App\Models\ComputerProduct;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ComputerProductController extends Controller
{
    public function index()
    {
        $computers = ComputerProduct::latest()->get();
        return view('computers.index', compact('computers'));
    }

    public function create()
    {
        return view('computers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'description' => ['nullable', 'string', 'max:1000'],
            'code_name' => ['required', 'string', 'max:50'],
            'name' => ['required', 'string', 'max:255'],
            'display' => ['required', 'numeric'],
            'cpu' => ['required', 'string', 'max:50'],
            'capacity' => ['required', 'regex:/^\\d+C\\/\\d+G$/i'],
            'ram' => ['required', 'regex:/^\\d+GB$/i'],
            'storage' => ['required', 'regex:/^\\d+GB$/i'],
            'color_code' => ['required', 'string', 'max:10', 'regex:/^[A-Za-z0-9]+$/'],
            'color' => ['required', 'string', 'max:50'],
            'country_code' => ['required', 'string', 'max:10', 'regex:/^[A-Za-z0-9]+$/'],
            'country' => ['required', 'string', 'max:50'],
            'condition' => ['required', 'regex:/^(NEW|USED|REFURBISHED)$/i'],
            'activate' => ['required', 'regex:/^(YES|NO|NA)$/i'],
            'internal_id' => ['required', 'string', 'max:255'],
            'cost' => ['nullable', 'numeric', 'min:0'],
            'tax' => ['nullable', 'numeric', 'min:0'],
            'stock_qty' => ['nullable', 'integer', 'min:0'],
        ]);

        $validated = $this->normalizePayload($validated);
        $validated['description'] = $this->buildDescription($validated);
        $validated['cost'] = $validated['cost'] ?? 0;
        $validated['tax'] = $validated['tax'] ?? 0;
        $validated['stock_qty'] = $validated['stock_qty'] ?? 0;
        $validated['product_code'] = $this->buildProductId($validated);
        $validated['serial_number'] = $validated['internal_id'];
        $validated['status'] = $validated['activate'] === 'YES' ? 'Active' : 'Inactive';

        if (ComputerProduct::where('product_code', $validated['product_code'])->exists()) {
            throw ValidationException::withMessages([
                'internal_id' => 'ProductID already exists.',
            ]);
        }

        $computer = ComputerProduct::create($validated);

        if ($request->expectsJson()) {
            return response()->json(['data' => $computer->fresh()], 201);
        }

        return redirect()->route('computers.index')->with('success', 'Computer product created.');
    }

    public function show(ComputerProduct $computer)
    {
        return view('computers.show', compact('computer'));
    }

    public function edit(ComputerProduct $computer)
    {
        return view('computers.edit', compact('computer'));
    }

    public function update(Request $request, ComputerProduct $computer)
    {
        $validated = $request->validate([
            'description' => ['nullable', 'string', 'max:1000'],
            'code_name' => ['required', 'string', 'max:50'],
            'name' => ['required', 'string', 'max:255'],
            'display' => ['required', 'numeric'],
            'cpu' => ['required', 'string', 'max:50'],
            'capacity' => ['required', 'regex:/^\\d+C\\/\\d+G$/i'],
            'ram' => ['required', 'regex:/^\\d+GB$/i'],
            'storage' => ['required', 'regex:/^\\d+GB$/i'],
            'color_code' => ['required', 'string', 'max:10', 'regex:/^[A-Za-z0-9]+$/'],
            'color' => ['required', 'string', 'max:50'],
            'country_code' => ['required', 'string', 'max:10', 'regex:/^[A-Za-z0-9]+$/'],
            'country' => ['required', 'string', 'max:50'],
            'condition' => ['required', 'regex:/^(NEW|USED|REFURBISHED)$/i'],
            'activate' => ['required', 'regex:/^(YES|NO|NA)$/i'],
            'internal_id' => ['required', 'string', 'max:255'],
            'cost' => ['required', 'numeric', 'min:0'],
            'tax' => ['required', 'numeric', 'min:0'],
            'stock_qty' => ['required', 'integer', 'min:0'],
        ]);

        $validated = $this->normalizePayload($validated);
        $validated['description'] = $this->buildDescription($validated);
        $validated['status'] = $validated['activate'] === 'YES' ? 'Active' : 'Inactive';
        $validated['serial_number'] = $computer->serial_number ?: $validated['internal_id'];

        $computer->update($validated);

        return redirect()->route('computers.show', $computer)->with('success', 'Computer product updated.');
    }

    public function destroy(ComputerProduct $computer)
    {
        $computer->delete();
        return redirect()->route('computers.index')->with('success', 'Computer product deleted.');
    }

    public function import(Request $request)
    {
        $request->validate([
            'import_file' => 'required|file|mimetypes:text/plain,text/csv,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);

        $file = $request->file('import_file');

        $rows = $this->parseImportFile($file->getRealPath(), $file->getClientOriginalExtension());

        $header = array_shift($rows);
        if (! $header) {
            return back()->withErrors(['import_file' => 'The uploaded file is empty.']);
        }

        $header = array_map(fn ($value) => Str::of($value)->trim()->snake()->value(), $header);
        $headerMap = [
            'description' => 'description',
            'codename' => 'code_name',
            'code_name' => 'code_name',
            'product_code' => 'code_name',
            'internal_id' => 'internal_id',
            'internalid' => 'internal_id',
            'i_d' => 'internal_id',
            'id' => 'internal_id',
            'serial_number' => 'internal_id',
            'product_name' => 'product_name',
            'name' => 'product_name',
            'display' => 'display',
            'c_p_u' => 'cpu',
            'cpu' => 'cpu',
            'capacity' => 'capacity',
            'r_a_m' => 'ram',
            'ram' => 'ram',
            'storage' => 'storage',
            'colorcode' => 'color_code',
            'color_code' => 'color_code',
            'color' => 'color',
            'countrycode' => 'country_code',
            'country_code' => 'country_code',
            'country' => 'country',
            'condition' => 'condition',
            'activate' => 'activate',
            'status' => 'activate',
        ];
        $header = array_map(fn ($value) => $headerMap[$value] ?? $value, $header);
        $requiredHeaders = [
            'code_name',
            'product_name',
            'display',
            'cpu',
            'capacity',
            'ram',
            'storage',
            'color_code',
            'color',
            'country_code',
            'country',
            'condition',
            'activate',
            'internal_id',
        ];
        if ($header !== $requiredHeaders) {
            return back()->withErrors([
                'import_file' => 'Headers must be in this exact order: CodeName, Product Name, Display, CPU, Capacity, RAM, Storage, ColorCode, Color, CountryCode, Country, Condition, Activate, ID.',
            ]);
        }

        $rowsImported = 0;
        $errors = [];
        $maxErrors = 25;
        $seenProductCodes = [];
        $payloads = [];

        foreach ($rows as $index => $row) {
            if (count(array_filter($row)) === 0) {
                continue;
            }

            $data = array_combine($header, $row);
            if (! $data) {
                continue;
            }

            $payload = [
                'description' => trim((string) ($data['description'] ?? '')),
                'code_name' => trim((string) ($data['code_name'] ?? '')),
                'internal_id' => trim((string) ($data['internal_id'] ?? '')),
                'name' => trim((string) ($data['product_name'] ?? '')),
                'display' => trim((string) ($data['display'] ?? '')),
                'cpu' => trim((string) ($data['cpu'] ?? '')),
                'capacity' => trim((string) ($data['capacity'] ?? '')),
                'ram' => trim((string) ($data['ram'] ?? '')),
                'storage' => trim((string) ($data['storage'] ?? '')),
                'color_code' => trim((string) ($data['color_code'] ?? '')),
                'color' => trim((string) ($data['color'] ?? '')),
                'country_code' => trim((string) ($data['country_code'] ?? '')),
                'country' => trim((string) ($data['country'] ?? '')),
                'condition' => strtoupper(trim((string) ($data['condition'] ?? ''))),
                'activate' => strtoupper(trim((string) ($data['activate'] ?? ''))),
                'cost' => 0,
                'tax' => 0,
                'stock_qty' => 0,
            ];

            $rowNumber = $index + 2;
            $rowValidator = validator($payload, [
                'description' => ['nullable', 'string', 'max:1000'],
                'code_name' => ['required', 'string', 'max:50'],
                'internal_id' => ['required', 'string', 'max:255'],
                'name' => ['required', 'string', 'max:255'],
                'display' => ['required', 'numeric'],
                'cpu' => ['required', 'string', 'max:50'],
                'capacity' => ['required', 'regex:/^\\d+C\\/\\d+G$/i'],
                'ram' => ['required', 'regex:/^\\d+GB$/i'],
                'storage' => ['required', 'regex:/^\\d+GB$/i'],
                'color_code' => ['required', 'string', 'max:10', 'regex:/^[A-Za-z0-9]+$/'],
                'color' => ['required', 'string', 'max:50'],
                'country_code' => ['required', 'string', 'max:10', 'regex:/^[A-Za-z0-9]+$/'],
                'country' => ['required', 'string', 'max:50'],
                'condition' => ['required', 'regex:/^(NEW|USED|REFURBISHED)$/i'],
            'activate' => ['required', 'regex:/^(YES|NO|NA)$/i'],
            ]);

            if ($rowValidator->fails()) {
                $errors[] = "Row {$rowNumber}: " . implode(' ', $rowValidator->errors()->all());
                if (count($errors) >= $maxErrors) {
                    $errors[] = 'Too many errors. Showing the first ' . $maxErrors . ' rows only.';
                    break;
                }
                continue;
            }

            $payload = $this->normalizePayload($payload);
            $payload['description'] = $this->buildDescription($payload);
            $payload['product_code'] = $this->buildProductId($payload);
            $payload['serial_number'] = $payload['internal_id'];
            $payload['status'] = $payload['activate'] === 'YES' ? 'Active' : 'Inactive';

            if (isset($seenProductCodes[$payload['product_code']])) {
                $errors[] = "Row {$rowNumber}: Duplicate ProductID {$payload['product_code']} in import file.";
                if (count($errors) >= $maxErrors) {
                    $errors[] = 'Too many errors. Showing the first ' . $maxErrors . ' rows only.';
                    break;
                }
                continue;
            }

            if (ComputerProduct::where('product_code', $payload['product_code'])->exists()) {
                $errors[] = "Row {$rowNumber}: ProductID {$payload['product_code']} already exists.";
                if (count($errors) >= $maxErrors) {
                    $errors[] = 'Too many errors. Showing the first ' . $maxErrors . ' rows only.';
                    break;
                }
                continue;
            }

            $seenProductCodes[$payload['product_code']] = true;
            $payloads[] = $payload;
        }

        if (! empty($errors)) {
            throw ValidationException::withMessages([
                'import_file' => implode(' ', $errors),
            ]);
        }

        DB::transaction(function () use (&$rowsImported, $payloads) {
            foreach ($payloads as $payload) {
                ComputerProduct::create($payload);
                $rowsImported++;
            }
        });

        if ($request->expectsJson()) {
            return response()->json(['imported' => $rowsImported], 200);
        }

        return back()->with('success', "Imported {$rowsImported} computer products.");
    }

    public function search(Request $request)
    {
        $query = trim((string) $request->get('query', ''));
        if ($query === '') {
            return response()->json(['data' => []]);
        }

        $like = '%' . $query . '%';
        $results = ComputerProduct::query()
            ->where(function ($builder) use ($like) {
                $builder->where('product_code', 'like', $like)
                    ->orWhere('name', 'like', $like)
                    ->orWhere('internal_id', 'like', $like)
                    ->orWhere('code_name', 'like', $like);
            })
            ->orderBy('product_code')
            ->limit(20)
            ->get([
                'id',
                'product_code',
                'code_name',
                'name',
                'display',
                'cpu',
                'capacity',
                'ram',
                'storage',
                'color_code',
                'color',
                'country_code',
                'country',
                'condition',
                'activate',
                'internal_id',
                'description',
                'serial_number',
                'cost',
            ]);

        return response()->json(['data' => $results]);
    }

    protected function parseImportFile(string $path, ?string $extension): array
    {
        $extension = strtolower($extension ?? '');

        if ($extension === 'csv' || str_contains(mime_content_type($path), 'csv')) {
            return $this->parseCsv($path);
        }

        if (in_array($extension, ['xlsx', 'xls'])) {
            if (! class_exists(IOFactory::class)) {
                throw ValidationException::withMessages([
                    'import_file' => 'Excel imports require phpoffice/phpspreadsheet. Please install it or upload a CSV file.',
                ]);
            }

            $reader = IOFactory::createReader('Xlsx');
            $spreadsheet = $reader->load($path);
            $worksheet = $spreadsheet->getActiveSheet();
            return $worksheet->toArray();
        }

        throw ValidationException::withMessages([
            'import_file' => 'Unsupported file type. Please upload a CSV or Excel (.xlsx) file.',
        ]);
    }

    protected function parseCsv(string $path): array
    {
        $rows = [];
        if (($handle = fopen($path, 'r')) !== false) {
            while (($row = fgetcsv($handle, 0, ',')) !== false) {
                $rows[] = $row;
            }
            fclose($handle);
        }

        return $rows;
    }

    protected function normalizePayload(array $payload): array
    {
        $payload['color_code'] = strtoupper(trim((string) ($payload['color_code'] ?? '')));
        $payload['country_code'] = strtoupper(trim((string) ($payload['country_code'] ?? '')));
        $payload['condition'] = strtoupper(trim((string) ($payload['condition'] ?? '')));
        $payload['activate'] = strtoupper(trim((string) ($payload['activate'] ?? '')));

        return $payload;
    }

    protected function buildProductId(array $payload): string
    {
        $parts = [
            trim((string) ($payload['internal_id'] ?? '')),
            trim((string) ($payload['ram'] ?? '')),
            trim((string) ($payload['storage'] ?? '')),
            strtoupper(trim((string) ($payload['color_code'] ?? ''))),
            strtoupper(trim((string) ($payload['country_code'] ?? ''))),
            trim((string) ($payload['capacity'] ?? '')),
        ];

        return implode('-', $parts);
    }

    protected function buildDescription(array $payload): string
    {
        $productName = trim((string) ($payload['name'] ?? ''));
        $display = trim((string) ($payload['display'] ?? ''));
        $cpu = trim((string) ($payload['cpu'] ?? ''));
        $capacity = trim((string) ($payload['capacity'] ?? ''));
        $ram = trim((string) ($payload['ram'] ?? ''));
        $storage = trim((string) ($payload['storage'] ?? ''));
        $color = trim((string) ($payload['color'] ?? ''));
        $country = trim((string) ($payload['country'] ?? ''));
        $condition = trim((string) ($payload['condition'] ?? ''));
        $activate = trim((string) ($payload['activate'] ?? ''));

        $main = trim(implode(' ', array_filter([$productName, $display])));
        $specs = trim(implode(' ', array_filter([
            $cpu,
            $capacity,
            $ram,
            $storage,
            $color,
            $country,
            $condition,
            $activate,
        ])));

        if ($main === '') {
            return $specs;
        }
        if ($specs === '') {
            return $main;
        }

        return $main . ' - ' . $specs;
    }
}
